<?php

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'Pwf_Filter_Manager' ) ) {

	/**
	 * Manage Filter on the frontend
	 *
	 * This is one place to control all related to filter
	 * Use to fix if the filter items code start before post code
	 *
	 * @ since 1.6.5
	 */
	class Pwf_Filter_Manager {

		public function __construct() {}

		/**
		 * Make apply filter id at one place
		 */
		public static function get_filter_id( $filter_id ) {
			return absint( apply_filters( 'pwf_filter_id', $filter_id ) );
		}

		/**
		 * Use on many classes
		 *
		 * @since 1.6.4
		 *
		 * @return string|empty prefixed string
		 */
		public static function get_pretty_urls_prefixed() {
			return get_option( 'pwf_shop_pretty_urls_prefixed', '' );
		}

		/**
		 * Set the global variable used in many places
		 * @param array $vars key => value
		 *
		 * @since 1.0.0, 1.6.5
		 */
		public static function set_pwf_global_variables( $vars = array() ) {
			if ( empty( $vars ) ) {
				$GLOBALS['pwf_main_query'] = array(
					'filter_id'         => '',
					'post_type'         => '',
					'query_vars'        => '',
					'filter_integrated' => 'no',
					'page_type'         => '', // Maybe archive|taxonomy|page|author  current_page_type
					'is_archive'        => 'false',
					'taxonomy_id'       => '',
					'taxonomy_name'     => '', // Example, product_cat
					'page_id'           => '',
					'user_id'           => '',
					'lang'              => '',
				);
			} else {
				foreach ( $vars as $key => $value ) {
					$GLOBALS['pwf_main_query'][ $key ] = $value;
				}
			}
		}

		/**
		 * Use this function only on frontend
		 * @param int $filter_id
		 *
		 * @since 1.6.5, 1.6.6
		 *
		 * return array ex array( 'items' => array(), 'setting' => array() )
		 */
		public static function get_filter_settings_and_items( $filter_id ) {

			return Pwf_Wpml::get_filter_post_meta( $filter_id );
		}

		/**
		 * Excute pwf shortcode && widget
		 * @param int $filter_id
		 *
		 * @since 1.6.5
		 *
		 * @return string HTML filter items
		 */
		public static function excute_plugin_shortcode_widget( $filter_id ) {
			$results      = '';
			$filter_id    = self::get_filter_id( $filter_id );
			$filter_data  = self::get_filter_settings_and_items( $filter_id );
			$filter_items = $filter_data['items'] ?? array();

			if ( ! empty( $filter_items ) ) {
				if ( ! empty( $GLOBALS['pwf_main_query']['query_vars'] ) ) {
					$query_vars = $GLOBALS['pwf_main_query']['query_vars'];
				} else {
					$selected_options = self::get_user_selected_options( $filter_id, $filter_items );
					$query_vars       = new Pwf_Parse_Query_Vars( $filter_id, $selected_options );

					$GLOBALS['pwf_main_query']['query_vars'] = $query_vars;
				}

				$render_filter = new Pwf_Render_Filter( $filter_id, $query_vars );
				$results       = $render_filter->get_html();
			}

			return $results;
		}

		/**
		 * Get all filter items inside filter post without columns or buttons
		 *
		 * @since 1.0.0, 1.6.5
		 *
		 * @return array
		 */
		public static function get_filter_items_without_columns( $filter_items ) {
			$items = array();
			foreach ( $filter_items as $item ) {
				if ( 'column' === $item['item_type'] ) {
					if ( ! empty( $item['children'] ) ) {
						$children = self::get_filter_items_without_columns( $item['children'] );
						$items    = array_merge( $items, $children );
					}
				} elseif ( 'button' !== $item['item_type'] ) {
					array_push( $items, $item );
				}
			}
			return $items;
		}

		/**
		 * Return selected option by user on frontend
		 *
		 * @param int $filter_id
		 * @param array $filter_items
		 *
		 * @since 1.6.5
		 *
		 * @return array selected options
		 */
		public static function get_user_selected_options( $filter_id, $filter_items ) {
			$selected_options = array();
			$filter_items     = self::get_filter_items_without_columns( $filter_items );

			if ( empty( $filter_items ) ) {
				return $selected_options;
			}

			if ( isset( $GLOBALS['pwf_main_query']['selected_options'] ) ) {
				$selected_options = self::get_selected_options_from_pretty_links( $filter_items );
			} else {
				// check what item is active
				foreach ( $filter_items as $item ) {
					if ( 'priceslider' === $item['item_type'] && 'two' === $item['price_url_format'] ) {
						if ( isset( $_GET[ $item['url_key_min_price'] ] ) && isset( $_GET[ $item['url_key_max_price'] ] ) ) {
							$selected_options[ $item['url_key'] ] = array( $_GET[ $item['url_key_min_price'] ], $_GET[ $item['url_key_max_price'] ] );
						}
					} elseif ( 'rangeslider' === $item['item_type'] && 'two' === $item['range_slider_url_format'] ) {
						if ( isset( $_GET[ $item['url_key_range_slider_min'] ] ) && isset( $_GET[ $item['url_key_range_slider_max'] ] ) ) {
							$selected_options[ $item['url_key'] ] = array( $_GET[ $item['url_key_range_slider_min'] ], $_GET[ $item['url_key_range_slider_max'] ] );
						}
					} elseif ( 'date' === $item['item_type'] ) {
						if ( isset( $_GET[ $item['url_key_date_after'] ] ) && isset( $_GET[ $item['url_key_date_before'] ] ) ) {
							$selected_options[ $item['url_key'] ] = array( $_GET[ $item['url_key_date_after'] ], $_GET[ $item['url_key_date_before'] ] );
						}
					} elseif ( isset( $_GET[ $item['url_key'] ] ) ) {
						$selected_options[ $item['url_key'] ] = $_GET[ $item['url_key'] ];
					}
				}
			}

			return $selected_options;
		}

		/**
		 * Get selected options if we use pretty links
		 * @see class pwf_parse_request
		 *
		 * @param array $filter_items
		 *
		 * @since 1.5.7
		 *
		 * @return array validated selected options
		 */
		protected static function get_selected_options_from_pretty_links( $filter_items ) {
			$result           = array();
			$selected_options = $GLOBALS['pwf_main_query']['selected_options'];
			foreach ( $filter_items as $item ) {
				$url_key = $item['url_key'];
				if ( isset( $selected_options[ $url_key ] ) ) {
					$result[ $url_key ] = $selected_options[ $url_key ];
				}
			}

			return $result;
		}
	}
}
