<?php

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'Pwf_Parse_Request' ) ) {

	/**
	 * Use for pretty links
	 * Parse Request URL and remove url keys form its if exists
	 * Return Main URL without url_keys and its values
	 *
	 * @since 1.5.7
	 */

	class Pwf_Parse_Request {

		protected static $prefixed;

		public function __construct() {
			if ( self::is_parse() ) {
				add_filter( 'pwf_routing_add_routes', array( $this, 'add_routes' ), 10, 2 );
				add_filter( 'do_parse_request', array( $this, 'do_parse_request' ), 10, 3 );
			}
		}

		public static function is_parse() {
			return ( ! is_admin() || ( defined( 'DOING_AJAX' ) && DOING_AJAX ) );
		}

		/**
		 * Check if the URL start with prefixed.
		 * @since 1.6.2
		 *
		 * @return string prefixed or empty
		 */
		public static function get_url_prefixed() {
			$prefixed = Pwf_Filter_Manager::get_pretty_urls_prefixed();

			if ( ! empty( $prefixed ) ) {
				$prefixed = sanitize_key( $prefixed ) . '/';
			}

			return $prefixed;
		}

		protected static function set_url_prefixed() {
			self::$prefixed = self::get_url_prefixed();
		}

		public function do_parse_request( $do_parse, $wp, $extra_query_vars ) {
			$current_url = self::get_current_url();
			if ( empty( $current_url ) || '?wc-ajax=get_refreshed_fragments' === $current_url ) {
				return $do_parse;
			}

			$current_url = self::remove_get_query_string( $current_url );

			if ( empty( $current_url ) ) {
				return $do_parse;
			}

			$routes = apply_filters( 'pwf_routing_add_routes', array(), $current_url );

			if ( empty( $routes ) || ! is_array( $routes ) ) {
				return $do_parse;
			}

			self::set_url_prefixed();

			$query_vars = self::parse_routes( $routes, $current_url );

			// If parse_routes() returns an array of query arguments as we expect...
			if ( ! empty( $query_vars ) && is_array( $query_vars ) ) {
				self::is_server_uri_requires_change( $query_vars );
				self::set_selected_options( $query_vars );
			} else {
				self::remove_prefixed_if_exists();
			}

			return $do_parse;
		}

		/**
		 * Get current page url.
		 * eg shop/product-catgory-clothing, shop/product-catgory-clothing/?min-price=50
		 *
		 * @since 1.5.9, 1.6.2
		 */
		protected static function get_current_url() {
			// Get current URL path, stripping out slashes on boundaries
			$current_url = trim( esc_url_raw( add_query_arg( array() ) ), '/' );
			// Get the path of the home URL, stripping out slashes on boundaries
			$home_path = parse_url( home_url(), PHP_URL_PATH );
			if ( ! empty( $home_path ) ) {
				$home_path = trim( parse_url( home_url(), PHP_URL_PATH ), '/' );
			}

			// If a URL part exists, and the current URL part starts with it...
			if ( $home_path && strpos( $current_url, $home_path ) === 0 ) {
				// ... just remove the home URL path form the current URL path
				$current_url = trim( substr( $current_url, strlen( $home_path ) ), '/' );
			}

			return $current_url;
		}

		/**
		 * If the URLs have query string vars remove it.
		 * eg. ?preview=1&orderby=price
		 *
		 * @return string
		 */
		protected function remove_get_query_string( string $url ) {
			$url_parts = explode( '?', $url, 2 );
			return $url_parts[0];
		}

		/**
		 * Parse routes define by our plugin.
		 *
		 * @param array $routes
		 * @param string $url
		 *
		 * @return array nested array, eg array( array( 'orginal_url' => string, 'values' => array() ))
		 */
		protected static function parse_routes( $routes, $url ) {
			$query_vars = array();

			if ( ! empty( self::$prefixed ) ) {
				$check_prefixed = self::$prefixed;
				if ( strpos( $url, $check_prefixed ) === false ) {
					return $query_vars;
				}
			}

			$url_array = explode( '/', $url );

			foreach ( $routes as $route_key => $info ) {
				$check_key = $route_key . '-';

				// Because we unset array we don't need to continue if it is empty
				if ( empty( $url_array ) ) {
					break;
				}

				foreach ( $url_array as $index => $url_route ) {
					if ( substr( $url_route, 0, strlen( $check_key ) ) === $check_key ) {
						unset( $url_array[ $index ] );
						if ( strlen( $url_route ) > strlen( $check_key ) ) {
							$query_vars[ $route_key ] = self::get_query_var_from_route( $url_route, $route_key, $info );
						} else {
							/**
							 * This is uncomplete url filter, eg. /product-cat-(empty)/
							 * Prefer way keep it and make WordPress recognize it
							 */
						}
					}
				}
			}

			return $query_vars;
		}

		/**
		 * Extract data from url string
		 *
		 * @param string $url the custom URL add to browser link
		 * @param string $url_key
		 * @param array $info is $url_key_info contain data like split
		 *
		 * @return array $query_var
		 */
		protected static function get_query_var_from_route( string $url, string $url_key, array $info ) {
			$query_var['orginal_url'] = $url;

			$split = $url_key . '-';
			$url   = str_replace( $split, '', $url );

			if ( empty( $info['separator'] ) ) {
				$values = array( esc_attr( $url ) );
			} elseif ( 'for' === $info['separator'] ) {
				$pattern = $info['separator'] . '-';
				$url     = str_replace( $pattern, '', $url );
				$value   = str_replace( '-', ' ', $url ); // (-) Is a white space
				$values  = array( $value );
			} else {
				$pattern = '-' . $info['separator'] . '-';
				$values  = explode( $pattern, $url );
			}

			$query_var['values'] = $values;

			return $query_var;
		}

		/**
		 * Set a server REQUEST_URI if prefixed is empty
		 *
		 * @since 1.6.4
		 */
		protected static function set_request_server_uri( $current_url ) {
			$_SERVER['REQUEST_URI'] = $current_url;
		}

		/**
		 * Check if the server REQUEST_URI requies changing.
		 * Use when prefixed is empty
		 * By remove all PWF plugin url_keys from the main URL
		 *
		 *  @param array $query_vars
		 *
		 * @since 1.6.4
		 *
		 * @return none.
		 */
		protected static function is_server_uri_requires_change( $query_vars ) {

			//if ( empty( self::$prefixed ) ) {
			$current_url = $_SERVER['REQUEST_URI'];
			$remove_url  = array_column( $query_vars, 'orginal_url' );

			/**
			 * Don't remove page/s number
			 */
			foreach ( $remove_url as $url ) {
				$count       = 0;
				$current_url = str_replace( $url . '/', '', $current_url, $count );
				if ( 0 === $count ) {
					error_log( 'the plugin can\'t remove the URL string ' . $url );
				}
			}

			/**
			 * because add prefixed doesn't work with shop archive and custom taxonmies
			 * If solved we can rmove this code
			 *
			 * need to check if prefixed exists in string before we remove it
			 */
			if ( ! empty( self::$prefixed ) ) {
				$remove_prefixed = self::$prefixed;

				$current_url = str_replace( $remove_prefixed, '', $current_url, $count );
				if ( 0 === $count ) {
					error_log( 'the plugin can\'t remove the prefixed string ' . $url );
				}
			}

			self::set_request_server_uri( $current_url );
			//}
		}

		/**
		 * Remove prefixed if exists and
		 *
		 * This function doesn't change browser url we need to redirect the site to current page without the prefixed.
		 * if url has prefixed doesn't remove it from browser url
		 * but this function removes prefixed from WP code sql.
		 * we can redirect site to current link 'feature'
		 *
		 * @since 1.6.4
		 */
		protected static function remove_prefixed_if_exists() {
			if ( ! empty( self::$prefixed ) ) {
				$current_url = $_SERVER['REQUEST_URI'];
				if ( strpos( $current_url, self::$prefixed ) !== false ) {
					$url_parts    = explode( '?', $current_url, 2 );
					$query_string = ''; // if url has '?'

					if ( isset( $url_parts[1] ) ) {
						$query_string = '?' . $url_parts[1];
					}

					$current_url = substr( $current_url, 0, strpos( $current_url, self::$prefixed ) );

					self::set_request_server_uri( $current_url . $query_string );
				}
			}
		}

		/**
		 * Save selected options in global variable
		 */
		protected static function set_selected_options( $query_vars ) {
			$selected_options = array();
			foreach ( $query_vars as $key => $data ) {
				$selected_options[ $key ] = $data['values'];
			}

			$GLOBALS['pwf_main_query']['selected_options'] = $selected_options;
		}

		/**
		 * Get routes that used by our plugin
		 *
		 * @return array routes
		 */
		public function add_routes( $routes, $current_url ) {
			$routes = get_option( 'pwf_woo_filter_routes', null );
			if ( null === $routes ) {
				$routes = array();
			}

			return $routes;
		}
	}

	new Pwf_Parse_Request();
}
