<?php

defined( 'ABSPATH' ) || exit;

if ( ! class_exists( 'Pwf_Wpml' ) ) {

	/**
	 * Group All function that related to the plugin WPML at one place
	 *
	 * Add apply filters so for other translated plugins
	 *
	 * @ since 1.5.5, 1.6.6
	 */
	class Pwf_Wpml {

		public function __construct() {

		}

		/**
		 * Check if the wpml activatate
		 *
		 * @since 1.5.5, 1.6.6
		 *
		 * @return boolean
		 */
		public static function is_wpml_defined() {
			return function_exists( 'wpml_object_id_filter' );
		}

		/**
		 * Check if the WPML plugin is exists or any other plugins like WPML
		 *
		 * @since 1.6.6
		 *
		 * @return bool
		 */
		public static function is_site_using_translated_plugin() {
			$defined = self::is_wpml_defined();

			/**
			 * Check if website using multi language plugin.
			 *
			 * @param bool $defined
			 */
			return apply_filters( 'pwf_is_site_using_translated_plugin', $defined );
		}

		public static function get_language() {
			return apply_filters( 'wpml_current_language', null );
		}

		/**
		 * Get Current Language Site
		 *
		 * @since 1.6.6
		 *
		 * @return string language | empty
		 */
		public static function get_current_language() {
			$lang = '';

			if ( self::is_wpml_defined() ) {
				$lang = ICL_LANGUAGE_CODE;
			}

			return apply_filters( 'pwf_get_current_language', $lang );
		}

		/**
		 * Fix Multi Language website
		 * Get the transalted Filer ID if exists
		 *
		 * @param int $filter_id
		 *
		 * @since 1.5.5
		 *
		 * @return array meta data
		 */
		public static function get_filter_post_meta( $filter_id ) {
			$meta_data = array();
			$filter_id = absint( $filter_id );

			if ( self::is_wpml_defined() ) {
				$filter_id = apply_filters( 'wpml_object_id', $filter_id, 'pwf_woofilter', true, self::get_language() );
				$meta_data = get_post_meta( $filter_id, '_pwf_woo_post_filter', true );
			} else {
				$filter_id = apply_filters( 'pwf_get_transalted_filter_post_id', $filter_id );
				$meta_data = get_post_meta( $filter_id, '_pwf_woo_post_filter', true );
			}

			if ( ! is_array( $meta_data ) ) {
				$meta_data = array();
			}

			return $meta_data;
		}

		/**
		 * For Multi language used to convert orginal term to translated term
		 * useful for colorlist, checkbox list
		 *
		 * @param array $item filter item settings
		 * @param array $taxonomy_name Taxonomy name. For example, product_cat|prdouct_tag|...
		 * @param string $meta_key name can be colos or boxlistlabel
		 *
		 * @since 1.5.5, 1.6.6
		 *
		 * @return array
		 */
		public static function convert_term_ids_to_translated_term_ids( $item, $taxonomy_name, $meta_key ) {
			if ( self::is_wpml_defined() ) {
				foreach ( $item[ $meta_key ] as $key => $term ) {
					$item[ $meta_key ][ $key ]['term_id'] = self::get_translated_term_id( $term['term_id'], $taxonomy_name );
				}
			}

			return $item;
		}

		/**
		 * For Multi language used to convert orginal include or exclude term IDs to translated term IDs
		 * Convert Include/Exclude term IDS to translated term IDs
		 * for include & exclude terms
		 *
		 * @param array $term_ids Term Ids need to get translated term IDS
		 * @param string $taxonomy_name Taxonomy name. For example, product_cat|prdouct_tag|...
		 *
		 * @since 1.5.5, 1.6.6
		 *
		 * @return array
		 */
		public static function convert_include_term_ids_to_translated_term_ids( $term_ids, $taxonomy_name ) {
			if ( self::is_wpml_defined() ) {
				$result = array();
				foreach ( $term_ids as $id ) {
					$id = self::get_translated_term_id( $id, $taxonomy_name );
					array_push( $result, $id );
				}
			} else {
				$result = $term_ids;
			}

			return $result;
		}

		/**
		 * Get translated term id
		 *
		 * @param int $term_id
		 * @param string term_type Maybe product_cat, product_tag, page, any taxonomy_name ..
		 *
		 * @since 1.5.5, 1.6.6
		 *
		 * @return int translated term ID
		 */
		public static function get_translated_term_id( $term_id, $term_type ) {
			if ( self::is_wpml_defined() ) {
				$term_id = apply_filters( 'wpml_object_id', $term_id, $term_type, true, self::get_language() );
			} else {
				/**
				 * Use to make other WordPress translated Plugins works like WPML plugin
				 *
				 * @param int $term_id can page_id or taxonomy ID
				 * @param string $term_type Maybe product_cat|prdouct_tag|taxonomy_name|..
				 */
				$term_id = apply_filters( 'pwf_translated_term_id_to_current_language', $term_id, $term_type );
			}

			return $term_id;
		}

	}
}
